/** Doxygen Style Comments
\file main.c
\brief d.bearing module for plotting a transect from point (x,y) along a bearing (theta) CCW from north. Optionally saving the line to a line vector (output).
 
\author Dylan Beaudette
\date 2005.12.05

\todo Add some more error correction
\todo Add ticks along transect length, and nodes to output vector

\note Some example useage:
\n 
\n #print from next click
\n d.bearing origin=`d.where -1 | awk '{print $1","$2}'` theta=230 radius=500
\n
\n #print from point associated with cat = 222
\n d.bearing origin=`v.out.ascii pedons | awk -F"|" '{if($3 == 222){print $1","$2}}'` theta=230 radius=500
 */

#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <unistd.h>
#include "gis.h"
#include "Vect.h"
#include "dbmi.h"
#include "display.h"
#include "raster.h"
#include "symbol.h"

#include "glocale.h"
#include "colors.h"

#define PI  M_PI
#define MAIN
#include "options.h"
#include "local_proto.h"


struct Cell_head window ;

int 
main (int argc, char **argv)
{
	struct GModule *module;
	struct Option *opt0, *opt1, *opt2, *opt3, *out_opt /*, *opt4*/ ;
	struct Map_info Out;
	int R, G, B, color = 0;
	const int customcolor = MAX_COLOR_NUM + 1;
	double x,y, dx, dy, theta_rad, theta_deg, radius; 
	double pixel; /* pixel size */
	int sign_dx = 1, sign_dy = 1;
	
	/* testing */
	struct line_pnts *Points, *Points_to_save;
	struct line_cats *Cats;
	int ltype;
	
	/* Initialize the GIS calls */
	G_gisinit(argv[0]) ;

	module = G_define_module();
	module->description = _("Plot a line from (x,y) along bearing theta to distance radius. Optionally saving the plotted line as a vector defined in output.");
	
	opt0 = G_define_option() ;
	opt0->key        = "origin" ;
	opt0->type       = TYPE_STRING ;
	opt0->key_desc   = "easting,northing" ;
	opt0->multiple   = NO;
	opt0->required   = YES;
	opt0->description= _("Starting point.") ;
	
	opt1 = G_define_option() ;
	opt1->key        = "theta" ;
	opt1->type       = TYPE_STRING ;
	opt1->required   = YES;
	opt1->description= _("degrees CCW from north, where north = 0 degrees.");
	
	opt2 = G_define_option() ;
	opt2->key        = "radius" ;
	opt2->type       = TYPE_STRING ;
	opt2->required   = NO;
	opt2->answer     = "100" ;
	opt2->description= "Length of transect in map units.";
	
	opt3 = G_define_option() ;
	opt3->key        = "color" ;
	opt3->type       = TYPE_STRING ;
	opt3->required   = NO;
	opt3->answer     = DEFAULT_FG_COLOR ;
	opt3->description= _("Color to draw with, either a standard GRASS color "
			   "or R:G:B triplet (separated by colons)");
	
	
	out_opt = G_define_standard_option(G_OPT_V_OUTPUT);
	out_opt->description   = "New vector line to store the transect output in." ;
	out_opt->required   = NO;


	/* Check command line */
	if (G_parser(argc, argv))
		exit(1);
	
	/* convert the theta answer(string) to double  */
	theta_deg = atof(opt1->answer);
	
	/* convert the answer(string) to double  */
	radius = atof(opt2->answer);
	
	/* extract the origin of the transect */
	x = atof(opt0->answers[0]);  /* 665413 */
	y = atof(opt0->answers[1]); /* 4038256 */
	
	
 	/* check theta value */
	if (opt1->answer != NULL)
 	{
 		/*check for a valid theta value*/
 		if(theta_deg < 0 || theta_deg > 360)
 			{
 			G_usage() ;
 			G_fatal_error ("Please enter a valid theta value in degrees CCW from north (0-360).");
 			}
 	}
	
	/* check radius value */
	if (opt2->answer != NULL)
 	{
 		/*check for a valid radius value*/
 		if(radius < 0)
 			{
 			G_usage() ;
 			G_fatal_error ("Please enter a valid radius value in.");
 			}
 	}

	/* Parse and select color */
	if (opt3->answer != NULL) {
	   color = G_str_to_color(opt3->answer, &R, &G, &B);
	   if(color == 0)
		G_fatal_error("[%s]: No such color", opt3->answer);
	   if(color == 1) {
		R_reset_color(R, G, B, customcolor);
		R_color(customcolor);
	   }
	   /* (color==2) is "none", noop */
	}



	vsize = hsize = 5.0 ;

	if (R_open_driver() != 0)
		G_fatal_error ("No graphics device selected");

	D_setup(0);

	G_get_set_window(&window) ;

	R_move_abs(
	    (int)(D_get_d_west() + D_get_d_east() / 2.0),
	    (int)(D_get_d_north() + D_get_d_south() / 2.0)) ;
	
	/*set_text_size() ;*/

	/* Do the graphics */
	G_setup_plot (
	    D_get_d_north(), D_get_d_south(), D_get_d_west(), D_get_d_east(),
	    D_move_abs, D_cont_abs);
	
	/* get the pixel size in map units */
	pixel = D_d_to_u_col(2) - D_d_to_u_col(1);
	/* printf("pixel = %f \n", pixel); */
	
	
	/* quadrant 1 */
	if(theta_deg >= 0 && theta_deg <= 90)
		{
		/* calculate the sign of dx and dy */
		sign_dx = -1;
		sign_dy = 1;	
		
		/* calculate our dx and dy values via simple trig  */
		theta_rad = theta_deg * PI/180 ;
		dx = sign_dx * radius * sin(theta_rad);
		dy = sign_dy * radius * cos(theta_rad);

		/* plot the line */
		G_plot_line (x, y, x+dx, y+dy);
		
		/* plot a pixel mark at the end of the line */
		G_plot_line (
		(x+dx)+(1 * 3 * pixel * cos((90 - (90 - theta_deg)) *(PI/180) ) ) , 
		(y+dy)+(1 * 3 * pixel * sin((90 - (90 - theta_deg)) *(PI/180) ) ) , 
		(x+dx)+(-1 * 3 * pixel * cos((90 - (90 - theta_deg)) *(PI/180) ) ) , 
		(y+dy)+(-1 * 3 * pixel * sin((90 - (90 - theta_deg)) *(PI/180) ) ) 
		);
		
		}
	
	/* quadrant 2 */		
	else if(theta_deg > 90 && theta_deg <= 180)
		{
		/* calculate the sign of dx and dy */
		sign_dx = -1;
		sign_dy = -1;

		/* adjust theta */
		theta_deg = theta_deg - 90;
	
		/* calculate our dx and dy values via simple trig  */
		theta_rad = theta_deg * PI/180 ;
		dx = sign_dx * radius * cos(theta_rad);
		dy = sign_dy * radius * sin(theta_rad);
	
		/* plot the line */
		G_plot_line (x, y, x+dx, y+dy);
		
		/* plot a pixel mark at the end of the line */
		G_plot_line (
		(x+dx)+(-1 * 3 * pixel * sin((90 - (90 - theta_deg)) *(PI/180) ) ) , 
		(y+dy)+(1 * 3 * pixel * cos((90 - (90 - theta_deg)) *(PI/180) ) ) , 
		(x+dx)+(1 * 3 * pixel * sin((90 - (90 - theta_deg)) *(PI/180) ) ) , 
		(y+dy)+(-1 * 3 * pixel * cos((90 - (90 - theta_deg)) *(PI/180) ) ) 
		);
		
		}	
	
	/* quadrant 3 */
	else if(theta_deg > 180 && theta_deg <= 270)
		{
		/* calculate the sign of dx and dy */
		sign_dx = 1;
		sign_dy = -1;

		/* adjust theta */
		theta_deg = theta_deg - 180;
		
		/* calculate our dx and dy values via simple trig  */
		theta_rad = theta_deg * PI/180 ;
		dx = sign_dx * radius * sin(theta_rad);
		dy = sign_dy * radius * cos(theta_rad);
		
		/* plot the line */
		G_plot_line (x, y, x+dx, y+dy);

		
		/* plot a pixel mark at the end of the line */
		G_plot_line (
		(x+dx)+(1 * 3 * pixel * cos((90 - (90 - theta_deg)) *(PI/180) ) ) , 
		(y+dy)+(1 * 3 * pixel * sin((90 - (90 - theta_deg)) *(PI/180) ) ) , 
		(x+dx)+(-1 * 3 * pixel * cos((90 - (90 - theta_deg)) *(PI/180) ) ) , 
		(y+dy)+(-1 * 3 * pixel * sin((90 - (90 - theta_deg)) *(PI/180) ) ) 
		);
		
		}

	/* quadrant 4 */
	else if(theta_deg > 270 && theta_deg <= 360)
		{
		/* calculate the sign of dx and dy */
		sign_dx = 1;
		sign_dy = 1;
		
		/* adjust theta */		
		theta_deg = theta_deg - 270;
		
		/* calculate our dx and dy values via simple trig  */
		theta_rad = theta_deg * PI/180 ;
		dx = sign_dx * radius * cos(theta_rad);
		dy = sign_dy * radius * sin(theta_rad);
		
		/* plot the line */
		G_plot_line (x, y, x+dx, y+dy);
		
		/* plot a pixel mark at the end of the line */
		G_plot_line (
		(x+dx)+(-1 * 3 * pixel * sin((90 - (90 - theta_deg)) *(PI/180) ) ) , 
		(y+dy)+(1 * 3 * pixel * cos((90 - (90 - theta_deg)) *(PI/180) ) ) , 
		(x+dx)+(1 * 3 * pixel * sin((90 - (90 - theta_deg)) *(PI/180) ) ) , 
		(y+dy)+(-1 * 3 * pixel * cos((90 - (90 - theta_deg)) *(PI/180) ) ) 
		);
		
		}
	
	
	 
// 	/* make a new vector object for drawing a box at the end of the line*/
// 	Points = Vect_new_line_struct();
// 	
// 	/* reset the vector starting pointer */
// 	Vect_reset_line ( Points );
// 	
// 	/* append some points to this vector object, in this case a square */
// 	Vect_append_point ( Points, (x+dx)+2*pixel, (y+dy)+2*pixel, 0);
// 	Vect_append_point ( Points, (x+dx)-2*pixel, (y+dy)+2*pixel, 0);
// 	Vect_append_point ( Points, (x+dx)-2*pixel, (y+dy)-2*pixel, 0);
// 	Vect_append_point ( Points, (x+dx)+2*pixel, (y+dy)-2*pixel, 0);
// 	
// 	/* plot the polygon */
// 	G_plot_polygon ( Points->x, Points->y, Points->n_points);
// 	
// 	/* clean up the temporary vector object  */
// 	Vect_destroy_line_struct (Points);
	
	
	if(out_opt->answer != NULL)
		{
		/*if the user has specified an 'output' then save the vector object */
		/* setup the new vector file */
		Vect_open_new(&Out, out_opt->answer, 0);
		
		/* make a new vector object storing the geometry of the line for later saving to a vector file*/
		Points_to_save = Vect_new_line_struct();
		Cats = Vect_new_cats_struct();
		
		/* reset the vector starting pointer */
		Vect_reset_line ( Points_to_save );
		/* add the starting and ending vertices */
		Vect_append_point ( Points_to_save, x, y, 0);
		Vect_cat_set(Cats, 1, 1);
		Vect_append_point ( Points_to_save, x+dx, y+dy, 0);
		Vect_cat_set(Cats, 1, 2);
			
		/* write the output vector line */
		ltype = 2;  /* lines */
		Vect_write_line(&Out, ltype, Points_to_save, Cats);
		
		/* build topology for output vector */
		Vect_build(&Out, stderr);
		/* close output vector */
		Vect_close(&Out);
		
		/* clean up the temporary vector object  */
		Vect_destroy_line_struct (Points_to_save);
		}
	
	/* make the graphics persistant */
	D_add_to_list(G_recreate_command()) ;
	/* close the graphic driver */
	R_close_driver();
	exit (0);
}
